//  -------------------------------------------------------------------------
/// \brief  Sequences editions
///
/// \author Logilin 2016-2022 - https://www.logilin.fr
///

	const ACTION_NOOP   = 0;
	const ACTION_RESET  = 1;
	const ACTION_RAMP   = 2;
	const ACTION_STEP   = 3;
	const ACTION_GOTO   = 4;
	const ACTION_REPEAT = 5;


/// \brief Step class constructor.
///
function Step(device_type, action, from, to, duration)
{
	this.action    = parseInt(action);
	this.from      = parseInt(from);
	this.to        = parseInt(to);
	this.duration  = parseInt(duration);

	switch (this.action) {
		case ACTION_NOOP:
		case ACTION_RESET:
			this.to  = 0;
			// no break;
		case ACTION_GOTO:
			this.from  = 0;
			break;
		case ACTION_REPEAT:
			break;
		case ACTION_RAMP:
			if (device_type == DEVICE_TYPE_SWITCH) {
				if (this.from >= Throws_per_switch)
					this.from = Throws_per_switch - 1;
			} else if (device_type == DEVICE_TYPE_ATTENUATOR) {
				if (this.from > Attenuators_range)
					this.from = Attenuators_range;
			} else if (device_type == DEVICE_TYPE_SPST) {
				if (this.from > 1)
					this.from = 1;
			} else {
				this.from = 0;
			}
			if (this.from < 0)
				this.from = 0;
			// no break;
		case ACTION_STEP:
			if (device_type == DEVICE_TYPE_SWITCH) {
				if (this.to >= Throws_per_switch)
					this.to = Throws_per_switch - 1;
			} else if (device_type == DEVICE_TYPE_ATTENUATOR) {
				if (this.to > Attenuators_range)
					this.to = Attenuators_range;
			} else if (device_type == DEVICE_TYPE_SPST) {
				if (this.to > 1)
					this.to = 1;
			} else {
				this.to = 0;
			}
			if (this.to < 0)
				this.to = 0;
			break;
	}
}



/// \brief Sequence class constructor
///
/// \param  filename     name of the data passed as second parameter.
/// \param  file_data    string containing all the file.
/// \param  device_type  0 for attenuators, 1 for switches, -1 means that the
///                      device type has to be determined from the data.
///
/// filename may be empty, resulting in a new sequence with a single "No-Op" step.
///
function Sequence(filename, file_data, device_type)
{
	this.steps = [];
	this.filename = filename;

	if (filename == "") {

		this.steps[0] = new Step(device_type, ACTION_NOOP, 0, 0, 0);
		this.device_type = device_type;

	} else {
		// `cgi-bin/load-sequence`` gives us a first line with the device type
		// then the normal step lines.

		var lines = file_data.replace("\r", "").split("\n");

		var regexp = new RegExp("[\t ]+", "g");

		if (lines[0].includes("switch"))
			this.device_type = DEVICE_TYPE_SWITCH;
		else if (lines[0].includes("spst"))
			this.device_type = DEVICE_TYPE_SPST;
		else
			this.device_type = DEVICE_TYPE_ATTENUATOR;

		for (var i = 1; i < lines.length; i++) {

			var fields = lines[i].split(regexp);

			if (fields.length >= 6) {
				this.steps[this.steps.length] = new Step(this.device_type, fields[2], fields[3], fields[4], fields[5]);
			}
		}

		// Remove leading No-Op steps.
		while ((this.steps.length > 1) && (this.steps[0].action == ACTION_NOOP))
			this.steps.shift();

		// Remove tailing No-Op steps.
		while ((this.steps.length > 1) && (this.steps[this.steps.length - 1].action == ACTION_NOOP))
			this.steps.pop();
	}
	this.saved = true;
	if ((this.device_type == DEVICE_TYPE_SWITCH) || (this.device_type == DEVICE_TYPE_SPST)) {
		if (localStorage.getItem('HytemLang') == "fr")
			document.getElementById("values-duration-granularity").innerHTML = "Durées en 1/100 sec.";
		else
			document.getElementById("values-duration-granularity").innerHTML = "Durations in 1/100 seconds.";
	} else {
		if (localStorage.getItem('HytemLang') == "fr")
			document.getElementById("values-duration-granularity").innerHTML = "Valeurs en 1/100 dB. Durées en 1/100 sec.";
		else
			document.getElementById("values-duration-granularity").innerHTML = "Values in 1/100 dB. Durations in 1/100 seconds.";
	}
}



/// \brief Prepare the content to send to the target.
///
Sequence.prototype.save_data = function()
{
	var i;
	var str = "";
	var timestamp = 0;

	// `t` and `n` will be replaced by `\t` and `\n` by `cgi-bin/save-sequence`.
	for (i = 0; i < this.steps.length; i++) {
		str += (i + 1)                + "t";
		str += timestamp              + "t";
		str += this.steps[i].action   + "t";
		str += this.steps[i].from     + "t";
		str += this.steps[i].to       + "t";
		str += this.steps[i].duration + "n";
		timestamp += parseInt(this.steps[i].duration);
	}
	return str;
}



/// \brief Fill the table with all the steps.
///
Sequence.prototype.fill_table = function()
{
	var i;

	var table = "";
	for (i = 0; i < this.steps.length; i ++) {
		table += "<tbody>";
		table += "<tr>";
		table +=   "<td class='edit-step-num'>" + (i+1) + "</td>";
		table +=   "<td class='edit-step-action'>";
		table +=     "<select class='edit-step-action-select' id='edit-step-action-select-" + i + "' onchange='edit_step_action_select_changed_cb(" + i +");' >";

		if (localStorage.getItem('HytemLang') == "fr") {

			table +=       "<option value='" + ACTION_NOOP + "'>Repos</option>";
			table +=       "<option value='" + ACTION_RESET + "'>Réinitialisation</option>";
			if ((this.device_type == DEVICE_TYPE_SWITCH) || (this.device_type == DEVICE_TYPE_SPST))
				table +=       "<option value='" + ACTION_RAMP + "' disabled>Rampe</option>";
			else
				table +=       "<option value='" + ACTION_RAMP + "'>Rampe</option>";
			table +=       "<option value='"+ ACTION_STEP + "'>Étape</option>";
			table +=       "<option value='" + ACTION_GOTO + "'>Saut</option>";
			table +=       "<option value='" + ACTION_REPEAT +"'>Répétition</option>";

		} else {

			table +=       "<option value='" + ACTION_NOOP + "'>No-Op</option>";
			table +=       "<option value='" + ACTION_RESET + "'>Reset</option>";
			if ((this.device_type == DEVICE_TYPE_SWITCH) || (this.device_type == DEVICE_TYPE_SPST))
				table +=       "<option value='" + ACTION_RAMP + "' disabled>Ramp</option>";
			else
				table +=       "<option value='" + ACTION_RAMP + "'>Ramp</option>";
			table +=       "<option value='"+ ACTION_STEP + "'>Step</option>";
			table +=       "<option value='" + ACTION_GOTO + "'>Goto</option>";
			table +=       "<option value='" + ACTION_REPEAT +"'>Repeat</option>";
		}

		table +=     "</select></td>";
		table +=   "<td class='edit-step-args' id='edit-step-label-1-" + i + "'></td>";
		table +=   "<td class='edit-step-args' id='edit-step-value-1-" + i + "'></td>";
		table +=   "<td class='edit-step-args' id='edit-step-label-2-" + i + "'></td>";
		table +=   "<td class='edit-step-args' id='edit-step-value-2-" + i + "'></td>";
		table +=   "<td class='edit-step-args' id='edit-step-label-3-" + i + "'></td>";
		table +=   "<td class='edit-step-args' id='edit-step-value-3-" + i + "'></td>";
		table +=   "<td class='edit-step-buttons'> <img class='edit-step-button' src='png/bt-add-step-above.png'  height=15 width=15 alt='Insert above' onclick='Current_sequence.insert_step_before(" + i + "); Current_sequence.fill_table();'/> </td>";
		table +=   "<td class='edit-step-buttons'> <img class='edit-step-button' src='png/bt-add-step-below.png'  height=15 width=15 alt='Insert below' onclick='Current_sequence.insert_step_after(" + i + ");  Current_sequence.fill_table();'/>  </td>";
		table +=   "<td class='edit-step-buttons'> <img class='edit-step-button' src='png/bt-del-step.png'        height=15 width=15 alt='Delete'       onclick='Current_sequence.remove_step(" + i + ");        Current_sequence.fill_table();'/>  </td>";
		table += "</tr>";
		table += "</tbody>";
	}
	document.getElementById('edit-sequence-table').innerHTML = table;

	for (i = 0; i < this.steps.length; i ++) {
		document.getElementById('edit-step-action-select-' + i).value = this.steps[i].action;
		this.display_line_arguments(i);
	}
	window_resize_cb();
}



/// \brief Insert a new step before the given one.
///
/// \param  num   The number of the current step.
///
Sequence.prototype.insert_step_before = function(num)
{
	var i;

	// Adjust the concerned Goto and Repeat.
	for (i = 0; i < this.steps.length; i ++)
		if ((this.steps[i].action == ACTION_GOTO) || (this.steps[i].action == ACTION_REPEAT))
			if (this.steps[i].to >= num)
				this.steps[i].to ++;

	// Insert a No-op step before num.
	this.steps.splice(num, 0, new Step(this.device_type, ACTION_NOOP, 0, 0, 0));

	something_modified();
	this.draw_canvas();
}



/// \brief Insert a new step  after the given one.
///
/// \param  num   The number of the current step.
///
Sequence.prototype.insert_step_after = function(num)
{
	var i;

	// Adjust the concerned Goto and Repeat.
	for (i = 0; i  < this.steps.length; i ++)
		if ((this.steps[i].action == ACTION_GOTO) || (this.steps[i].action == ACTION_REPEAT))
			if (this.steps[i].to > num)
				this.steps[i].to ++;

	// Insert a No-op step after num.
	this.steps.splice(num + 1, 0, new Step(this.device_type, ACTION_NOOP, 0, 0, 0));

	something_modified();
	this.draw_canvas();
}



/// \brief Remove a step from the sequence.
///
/// \param  num   The number of the current step.
///
Sequence.prototype.remove_step = function(num)
{
	if (this.steps.length <= 1)
		return;

	var i;

	// Adjust the concerned Goto and Repeat.
	for (i = 0; i  < this.steps.length; i ++)
		if ((this.steps[i].action == ACTION_GOTO) || (this.steps[i].action == ACTION_REPEAT))
			if (this.steps[i].to > num)
				this.steps[i].to --;

	// Remove the num step.
	this.steps.splice(num, 1);

	something_modified();
	this.draw_canvas();
}



/// \brief Fill the rest of the line with the correct argument depending of the selected action.
///
/// \param num  The concerned step number.
///
Sequence.prototype.display_line_arguments = function(num)
{
	var from     = this.steps[num].from;
	var to       = this.steps[num].to;
	var duration = this.steps[num].duration;

	var lang = localStorage.getItem('HytemLang');


	if (this.device_type == DEVICE_TYPE_SWITCH) {
		if ((this.steps[num].action == ACTION_RAMP) || (this.steps[num].action == ACTION_STEP)) {
			from ++;
			to ++;
		}
	}

	if (this.steps[num].action == ACTION_NOOP) {
		document.getElementById('edit-step-label-1-' + num).innerHTML = "";
		document.getElementById('edit-step-value-1-' + num).innerHTML = "";
		document.getElementById('edit-step-label-2-' + num).innerHTML = "";
		document.getElementById('edit-step-value-2-' + num).innerHTML = "";
		document.getElementById('edit-step-label-3-' + num).innerHTML = "";
		document.getElementById('edit-step-value-3-' + num).innerHTML = "";
	}

	if (this.steps[num].action == ACTION_RESET) {
		document.getElementById('edit-step-label-1-' + num).innerHTML = "";
		document.getElementById('edit-step-value-1-' + num).innerHTML = "";
		document.getElementById('edit-step-label-2-' + num).innerHTML = "";
		document.getElementById('edit-step-value-2-' + num).innerHTML = "";
		document.getElementById('edit-step-label-3-' + num).innerHTML = "";
		document.getElementById('edit-step-value-3-' + num).innerHTML = "";
	}

	if (this.steps[num].action == ACTION_RAMP) {
		if (lang == "fr")
			document.getElementById('edit-step-label-1-' + num).innerHTML = "De : ";
		else
			document.getElementById('edit-step-label-1-' + num).innerHTML = "From: ";
		document.getElementById('edit-step-value-1-' + num).innerHTML = "<input type='text' class='edit-step-value' id='edit-step-val-1-" + num + "' value='" + from +"' onchange='edit_step_val_1_changed_cb(" + num + ");' />";
		if (lang == "fr")
			document.getElementById('edit-step-label-2-' + num).innerHTML = "À : ";
		else
			document.getElementById('edit-step-label-2-' + num).innerHTML = "To: ";
		document.getElementById('edit-step-value-2-' + num).innerHTML = "<input type='text' class='edit-step-value' id='edit-step-val-2-" + num + "' value='" + to +"' onchange='edit_step_val_2_changed_cb(" + num + ");' />";
		if (lang == "fr")
			document.getElementById('edit-step-label-3-' + num).innerHTML = "Durée : ";
		else
			document.getElementById('edit-step-label-3-' + num).innerHTML = "Duration: ";
		document.getElementById('edit-step-value-3-' + num).innerHTML = "<input type='text' class='edit-step-value' id='edit-step-val-3-" + num + "' value='" + duration +"' onchange='edit_step_val_3_changed_cb(" + num + ");' />";
	}

	if (this.steps[num].action == ACTION_STEP) {
		document.getElementById('edit-step-label-1-' + num).innerHTML = "";
		document.getElementById('edit-step-value-1-' + num).innerHTML = "";
		if (lang == "fr")
			document.getElementById('edit-step-label-2-' + num).innerHTML = "Valeur : ";
		else
			document.getElementById('edit-step-label-2-' + num).innerHTML = "Value: ";
		document.getElementById('edit-step-value-2-' + num).innerHTML = "<input type='text' class='edit-step-value' id='edit-step-val-2-" + num + "' value='" + to +"' onchange='edit_step_val_2_changed_cb(" + num + ");' />";
		if (lang == "fr")
			document.getElementById('edit-step-label-3-' + num).innerHTML = "Durée : ";
		else
			document.getElementById('edit-step-label-3-' + num).innerHTML = "Duration: ";
		document.getElementById('edit-step-value-3-' + num).innerHTML = "<input type='text' class='edit-step-value' id='edit-step-val-3-" + num + "' value='" + duration +"' onchange='edit_step_val_3_changed_cb(" + num + ");' />";
	}

	if (this.steps[num].action == ACTION_GOTO) {
		document.getElementById('edit-step-label-1-' + num).innerHTML = "";
		document.getElementById('edit-step-value-1-' + num).innerHTML = "";
		if (lang == "fr")
			document.getElementById('edit-step-label-2-' + num).innerHTML = "Vers : ";
		else
			document.getElementById('edit-step-label-2-' + num).innerHTML = "To: ";
		document.getElementById('edit-step-value-2-' + num).innerHTML = "<input type='text' class='edit-step-value' id='edit-step-val-2-" + num + "' value='" + to +"' onchange='edit_step_val_2_changed_cb(" + num + ");' />";
		document.getElementById('edit-step-label-3-' + num).innerHTML = "";
		document.getElementById('edit-step-value-3-' + num).innerHTML = "";
	}

	if (this.steps[num].action == ACTION_REPEAT) {
		if (lang == "fr")
			document.getElementById('edit-step-label-1-' + num).innerHTML = "Nombre : ";
		else
			document.getElementById('edit-step-label-1-' + num).innerHTML = "Count: ";
		document.getElementById('edit-step-value-1-' + num).innerHTML = "<input type='text' class='edit-step-value' id='edit-step-val-1-" + num + "' value='" + from +"' onchange='edit_step_val_1_changed_cb(" + num + ");' />";
		if (lang == "fr")
			document.getElementById('edit-step-label-2-' + num).innerHTML = "Vers : ";
		else
			document.getElementById('edit-step-label-2-' + num).innerHTML = "To: ";
		document.getElementById('edit-step-value-2-' + num).innerHTML = "<input type='text' class='edit-step-value' id='edit-step-val-2-" + num + "' value='" + to +"' onchange='edit_step_val_2_changed_cb(" + num + ");' />";
		document.getElementById('edit-step-label-3-' + num).innerHTML = "";
		document.getElementById('edit-step-value-3-' + num).innerHTML = "";
	}
}



const MARGIN_HEIGHT = 20;

Sequence.prototype.draw_canvas = function()
{
	var step = 0; 
	var t = 0;
	var y_coef;

	if (this.steps.length <= 0)
		return;

	var ctx = cnv.getContext("2d");

	cnv.width  = cnv.clientWidth;
	cnv.height = cnv.clientHeight;

	if (this.device_type == DEVICE_TYPE_SWITCH)
		y_coef = (cnv.height - MARGIN_HEIGHT)/ (Throws_per_switch - 1);
	else if (this.device_type == DEVICE_TYPE_SPST)
		y_coef = cnv.height - MARGIN_HEIGHT;
	else
		y_coef = 1/100;

	// Horizontal rulers.
	ctx.beginPath();
	if (this.device_type == DEVICE_TYPE_SWITCH) {
		for (var i = 0; i < Throws_per_switch; i ++) {
			ctx.moveTo(0,         cnv.height - MARGIN_HEIGHT - i * y_coef);
			ctx.lineTo(cnv.width, cnv.height - MARGIN_HEIGHT - i * y_coef);
		}
	} else if (this.device_type == DEVICE_TYPE_SPST) {
		for (var i = 0; i < 2; i ++) {
			ctx.moveTo(0,         cnv.height - MARGIN_HEIGHT - i * y_coef);
			ctx.lineTo(cnv.width, cnv.height - MARGIN_HEIGHT - i * y_coef);
		}
	} else {
		for (var i = 0; i < 10000; i += 1000) {
			ctx.moveTo(0,         cnv.height - MARGIN_HEIGHT - i * y_coef);
			ctx.lineTo(cnv.width, cnv.height - MARGIN_HEIGHT - i * y_coef);
		}
	}
	ctx.strokeStyle = "rgba(0, 0, 0, 0.25)";
	ctx.stroke();

	// Vertical rulers.
	ctx.beginPath();
	t = 10 * parseInt(cnv.seq_start_t / 10);
	while ((t - cnv.seq_start_t) * cnv.seq_zoom < cnv.width) {
		ctx.moveTo((t  - cnv.seq_start_t) * cnv.seq_zoom,  0);
		ctx.lineTo((t  - cnv.seq_start_t) * cnv.seq_zoom,  cnv.height - MARGIN_HEIGHT);
		t += 10;
		if ((t % 600) == 0) {
			ctx.moveTo((t  - cnv.seq_start_t) * cnv.seq_zoom - 1,  0);
			ctx.lineTo((t  - cnv.seq_start_t) * cnv.seq_zoom - 1,  cnv.height - MARGIN_HEIGHT);
			ctx.moveTo((t  - cnv.seq_start_t) * cnv.seq_zoom + 1,  0);
			ctx.lineTo((t  - cnv.seq_start_t) * cnv.seq_zoom + 1,  cnv.height - MARGIN_HEIGHT);
		}
	}
	ctx.strokeStyle = "rgba(0, 0, 0, 0.25)";
	ctx.stroke();


	ctx.beginPath();
	ctx.moveTo(0, cnv.height - MARGIN_HEIGHT);

	for (var i = 0; i < this.steps.length; i ++)
		this.steps[i].counter = 0;

	t = 0;
	while (((t - cnv.seq_start_t) * cnv.seq_zoom < cnv.width) && (step < this.steps.length)) {
		if (this.steps[step].action == ACTION_NOOP) {
			step ++;
		} else if (this.steps[step].action == ACTION_RESET) {
			step ++;
		} else if (this.steps[step].action == ACTION_RAMP) {
			ctx.fillText(step+1, (t - cnv.seq_start_t) * cnv.seq_zoom, cnv.height - 2);
			ctx.lineTo((t - cnv.seq_start_t) * cnv.seq_zoom, cnv.height - MARGIN_HEIGHT - this.steps[step].from * y_coef);
			t += this.steps[step].duration;
			ctx.lineTo((t - cnv.seq_start_t) * cnv.seq_zoom, cnv.height - MARGIN_HEIGHT - this.steps[step].to * y_coef);
			step ++;
		} else if (this.steps[step].action == ACTION_STEP) {
			ctx.fillText(step+1, (t - cnv.seq_start_t) * cnv.seq_zoom, cnv.height - 2);
			ctx.lineTo((t - cnv.seq_start_t) * cnv.seq_zoom, cnv.height - MARGIN_HEIGHT - this.steps[step].to * y_coef);
			t += this.steps[step].duration;
			ctx.lineTo((t - cnv.seq_start_t) * cnv.seq_zoom, cnv.height - MARGIN_HEIGHT - this.steps[step].to * y_coef);
			step ++;
		} else if (this.steps[step].action == ACTION_GOTO) {
			step = this.steps[step].to - 1;
		} else if (this.steps[step].action == ACTION_REPEAT) {
			if (this.steps[step].counter < this.steps[step].from) {
				this.steps[step].counter++;
				step = this.steps[step].to - 1;
			} else {
				this.steps[step].counter = 0;
				step ++;
			}
		}
	}
	ctx.strokeStyle = "rgba(0, 0, 0, 1.0)";
	ctx.stroke();
}




	var is_mouse_down  = false;
	document.addEventListener('mouseup', function() {is_mouse_down = false;}, false);

	var cnv = document.getElementById("editor-graph-cnv");

	cnv.seq_mouse_x_prev = 0;
	cnv.seq_mouse_moving = false;
	cnv.seq_start_t      = 0;
	cnv.seq_zoom         = 2.0;

	cnv.onmousewheel = cnv_on_mouse_wheel; // I.E.
	if (cnv.addEventListener) // Mozilla
		cnv.addEventListener('DOMMouseScroll', cnv_on_mouse_wheel, false);



function cnv_on_mouse_wheel(evt)
{
	var mv;
	if (! evt)
		evt = window.event; // IE
	if (evt.wheelDelta) { // IE
		mv = - evt.wheelDelta / 40;
	} else if (evt.detail) { // Mozilla
		mv = evt.detail;
	}
	if (mv < 0) {
		if (cnv.seq_zoom < 50)
			cnv.seq_zoom *= 1.5;
	} else if (mv > 0) {
		if (cnv.seq_zoom > 0.05)
			cnv.seq_zoom /= 1.5;
	}
	if (evt.preventDefault)
		evt.preventDefault();
	evt.returnValue = false;
	Current_sequence.draw_canvas();
}



cnv.addEventListener('mousedown', function (evt)
	{
		cnv.seq_mouse_x_prev = evt.clientX;
		cnv.seq_mouse_moving = true;
		is_mouse_down = true;
	}, false);



cnv.addEventListener('mousemove', function (evt)
	{
		if (! is_mouse_down)
			cnv.seq_mouse_moving = false;
		if (! cnv.seq_mouse_moving)
			return;
		cnv.seq_start_t -= evt.clientX - cnv.seq_mouse_x_prev;
		if (cnv.seq_start_t < 0)
			cnv.seq_start_t = 0;
		cnv.seq_mouse_x_prev = evt.clientX;
		Current_sequence.draw_canvas();
	}, false);



cnv.addEventListener('mouseup', function (evt)
	{
		is_mouse_down = false;
		cnv.seq_mouse_moving = false;
		Current_sequence.draw_canvas();
	}, false);



function load_sequence(filename)
{
	var xhr = get_xhr_object();
	if (xhr == null)
		return;
	xhr.onreadystatechange = function() {
		if (xhr.readyState == 4) {
			if (xhr.status == 200) {
				Current_sequence = new Sequence(filename, xhr.responseText, DEVICE_TYPE_UNKNOWN);
				Current_sequence.fill_table();
				Current_sequence.draw_canvas();
				all_is_saved();
			}
		}
	};
	xhr.open("GET", "cgi-bin/load-sequence?"+filename, true);
	xhr.send();
}



function save_sequence(filename)
{
	if (Current_sequence != null) {
		var xhr = get_xhr_object();
		if (xhr == null)
			return;
		xhr.onreadystatechange = function() {
			if (xhr.readyState == 4) {
				if (xhr.status == 200) {
					all_is_saved();
				}
			}
		};
		var str = "cgi-bin/save-sequence?filename=" + filename + "&type=" + Current_sequence.device_type + "&content=" + Current_sequence.save_data();
		xhr.open("GET", str, true);
		xhr.send();
	}
}



function new_sequence()
{
	switch(Supported_devices) {
		case DEVICE_TYPE_ATTENUATOR:
			new_attenuator_sequence();
			break;
		case DEVICE_TYPE_SWITCH:
			new_switch_sequence();
			break;
		case DEVICE_TYPE_SPST:
			new_spst_sequence();
			break;
		default:
			device_type_selection_dialog.dialog("open");
			break;
	}
}



function new_attenuator_sequence()
{
	Current_sequence = new Sequence("", "", DEVICE_TYPE_ATTENUATOR);
	Current_sequence.fill_table();
	something_modified();
	Current_sequence.draw_canvas();
}



function new_switch_sequence()
{
	Current_sequence = new Sequence("", "", DEVICE_TYPE_SWITCH);
	Current_sequence.fill_table();
	something_modified();
	Current_sequence.draw_canvas();
}



function new_spst_sequence()
{
	Current_sequence = new Sequence("", "", DEVICE_TYPE_SPST);
	Current_sequence.fill_table();
	something_modified();
	Current_sequence.draw_canvas();
}



function edit_step_action_select_changed_cb(num)
{
	Current_sequence.steps[num].action = parseInt(document.getElementById('edit-step-action-select-' + num).value);
	Current_sequence.display_line_arguments(num);
	something_modified();
	Current_sequence.draw_canvas();
}



function edit_step_val_1_changed_cb(num)
{
	Current_sequence.steps[num].from = parseInt(document.getElementById('edit-step-val-1-' + num).value);

	if ((Current_sequence.steps[num].action == ACTION_STEP) || (Current_sequence.steps[num].action == ACTION_RAMP)) {

		if (Current_sequence.device_type == DEVICE_TYPE_ATTENUATOR) {

			if (Current_sequence.steps[num].from > Attenuators_range)
				Current_sequence.steps[num].from = Attenuators_range;
			if (Current_sequence.steps[num].from < 0)
				Current_sequence.steps[num].from = 0;

			document.getElementById('edit-step-val-1-' + num).value = Current_sequence.steps[num].from;

		} else if (Current_sequence.device_type == DEVICE_TYPE_SWITCH) {

			Current_sequence.steps[num].from --;

			if (Current_sequence.steps[num].from >= Throws_per_switch)
				Current_sequence.steps[num].from = Throws_per_switch - 1;
			if (Current_sequence.steps[num].from < 0)
				Current_sequence.steps[num].from = 0;

			document.getElementById('edit-step-val-1-' + num).value = Current_sequence.steps[num].from + 1;

		} else if (Current_sequence.device_type == DEVICE_TYPE_SPST) {

			if (Current_sequence.steps[num].from > 1)
				Current_sequence.steps[num].from = 1;
			if (Current_sequence.steps[num].from < 0)
				Current_sequence.steps[num].from = 0;

			document.getElementById('edit-step-val-1-' + num).value = Current_sequence.steps[num].from;
		}
	}

	something_modified();
	Current_sequence.draw_canvas();
}



function edit_step_val_2_changed_cb(num)
{
	Current_sequence.steps[num].to = parseInt(document.getElementById('edit-step-val-2-' + num).value);

	if ((Current_sequence.steps[num].action == ACTION_STEP) || (Current_sequence.steps[num].action == ACTION_RAMP)) {

		if (Current_sequence.device_type == DEVICE_TYPE_ATTENUATOR) {

			if (Current_sequence.steps[num].to > Attenuators_range)
				Current_sequence.steps[num].to = Attenuators_range;
			if (Current_sequence.steps[num].to < 0)
				Current_sequence.steps[num].to = 0;

		document.getElementById('edit-step-val-2-' + num).value = Current_sequence.steps[num].to;

		} else if (Current_sequence.device_type == DEVICE_TYPE_SWITCH) {

			Current_sequence.steps[num].to --;

			if (Current_sequence.steps[num].to >= Throws_per_switch)
				Current_sequence.steps[num].to = Throws_per_switch - 1;
			if (Current_sequence.steps[num].to < 0)
				Current_sequence.steps[num].to = 0;

			document.getElementById('edit-step-val-2-' + num).value = Current_sequence.steps[num].to + 1;

		} else if (Current_sequence.device_type == DEVICE_TYPE_SPST) {

			if (Current_sequence.steps[num].to > 1)
				Current_sequence.steps[num].to = 1;
			if (Current_sequence.steps[num].to < 0)
				Current_sequence.steps[num].to = 0;

			document.getElementById('edit-step-val-2-' + num).value = Current_sequence.steps[num].to;
		}
	}
	something_modified();
	Current_sequence.draw_canvas();
}



function edit_step_val_3_changed_cb(num)
{
	Current_sequence.steps[num].duration = parseInt(document.getElementById('edit-step-val-3-' + num).value);

	if (Current_sequence.steps[num].duration < 0)
		Current_sequence.steps[num].duration = 0;

	document.getElementById('edit-step-val-3-' + num).value = Current_sequence.steps[num].duration;
	something_modified();
	Current_sequence.draw_canvas();
}



function something_modified()
{
	if (Current_sequence.saved) {
		if (Current_sequence.filename == "")
			document.getElementById('edit-filename').innerHTML = "* (no name)";
		else
			document.getElementById('edit-filename').innerHTML = "* " + Current_sequence.filename;
	}
	Current_sequence.saved = false;
	window.onbeforeunload =  function(e) {
		e = e || window.event;
		// For IE and Firefox
		if (e) {
			e.returnValue = "Your modification will be lost";
		}
		return "Your modifications will be lost";
	};

}


function all_is_saved()
{
	Current_sequence.saved = true;
	if (Current_sequence.filename == "")
		document.getElementById('edit-filename').innerHTML = "  (no name)";
	else
		document.getElementById('edit-filename').innerHTML = "  " + Current_sequence.filename;
	window.onbeforeunload = function (e) { };
}
