//  --------------------------------------------------------------------------
///
/// \file config.h
//
/// \author Logilin - https://www.logilin.fr/
//
//  --------------------------------------------------------------------------

#ifndef CONFIG_H
#define CONFIG_H

	#include <stdint.h>

	/// \brief Initialize all the memory receiving data parameters.
	///
	/// Note: don't access any config parameters before calling this method.
	///
	/// \returns 0 on success, -1 on error.
	///
	int Config_init(void);


	/// \brief Free the parameters memory.
	///
	/// Note: don't access any config parameters after calling this method.
	///
	void Config_exit(void);


	/// \brief Set the debug level of this module.
	///
	/// \param level - the new debug level.
	///
	void Config_set_debug_level(int level);


	/// \brief Load a configuration file.
	///
	/// \param filename  The path + filename of the file to read.
	///
	/// \return 0 on success, -1 on error.
	///
	int Config_load_configuration (const char *filename);

	#define DEFAULT_DEVICES_CONFIG_FILE "/data/config/hytem-devices.config"


	// ----------------------------------------------------
	// GLOBAL PARAMETERS STRUCTURE

	typedef struct {

		// Parameters set during factory configuration.

		// 0
			#define NUMBER_OF_VERSION_VALUES  4 /// Major, Minor, Release, Type.
			#define VERSION_TYPE_RELEASE      0
			#define VERSION_TYPE_PRERELEASE   1
			#define VERSION_TYPE_DEVELOPMENT  2
		uint8_t version[NUMBER_OF_VERSION_VALUES];  ///< Software version.

		// 4

			#define MINIMAL_SUPPORTED_DEVICES    1
			#define MAXIMAL_SUPPORTED_DEVICES    7
			#define DEFAULT_SUPPORTED_DEVICES    1

		int32_t  supported_devices; ///< binary OR of supported device types.

		// 8

			#define MINIMAL_NUMBER_OF_GROUPS    1
			#define DEFAULT_NUMBER_OF_GROUPS    1
			#ifdef _MAXIMAL_NUMBER_OF_GROUPS
			  #define MAXIMAL_NUMBER_OF_GROUPS _MAXIMAL_NUMBER_OF_GROUPS
			#else
			  #define MAXIMAL_NUMBER_OF_GROUPS 16
			#endif
		int32_t  number_of_groups;   ///< Number of groups of devices.

		// 12
			#define MINIMAL_DEVICES_PER_GROUP     1
			#define DEFAULT_DEVICES_PER_GROUP     8
			#ifdef _MAXIMAL_DEVICES_PER_GROUP
			  #define MAXIMAL_DEVICES_PER_GROUP  _MAXIMAL_DEVICES_PER_GROUP
			#else
			  #define MAXIMAL_DEVICES_PER_GROUP  18
			#endif
		int32_t  devices_per_group;  ///< Number of devices per group.

		// 16
			#define MINIMAL_NUMBER_OF_DEVICES    (MINIMAL_NUMBER_OF_GROUPS * MINIMAL_DEVICES_PER_GROUP)
			#define DEFAULT_NUMBER_OF_DEVICES    (DEFAULT_NUMBER_OF_GROUPS * DEFAULT_DEVICES_PER_GROUP)
			#define MAXIMAL_NUMBER_OF_DEVICES    (MAXIMAL_NUMBER_OF_GROUPS * MAXIMAL_DEVICES_PER_GROUP)
			#ifdef _MAXIMAL_ALLOWED_NUMBER_OF_DEVICES
			  #define MAXIMAL_ALLOWED_NUMBER_OF_DEVICES  _MAXIMAL_ALLOWED_NUMBER_OF_DEVICES
			#else
			  #define MAXIMAL_ALLOWED_NUMBER_OF_DEVICES MAXIMAL_NUMBER_OF_DEVICES
			#endif
		int32_t  number_of_devices;  ///< Number of devices (redundant, to limit computation).

		// 20
			#define MINIMAL_ATTENUATOR_RANGE      6250
			#define DEFAULT_ATTENUATOR_RANGE      9350
			#define MAXIMAL_ATTENUATOR_RANGE     11000
		int32_t  attenuators_range;  ///< Attenuators range: 6250 (-62.5dB), 9350 (-93.5 dB) or 11000 (-110 dB).

		// 24
			#define MINIMAL_THROWS_PER_SWITCH         2
			#define DEFAULT_THROWS_PER_SWITCH         8
			#define MAXIMAL_THROWS_PER_SWITCH        16
		int32_t  throws_per_switch; ///<  Number of throws on Single Pole Multiple Throws switches.

		// 28
			#define PRODUCT_NUMBER_LENGTH_MAX    15
			#define DEFAULT_PRODUCT_NUMBER  "00.00.000"
		char product_number[PRODUCT_NUMBER_LENGTH_MAX + 1]; ///< Product number (nul terminated string).

		// 44
			#define SERIAL_NUMBER_LENGTH_MAX    15
			#define DEFAULT_SERIAL_NUMBER  "00.000.00"
		char serial_number[SERIAL_NUMBER_LENGTH_MAX + 1];   ///< Serial number (nul terminated string).

		// 60
			#define UPDATE_URL_LENGTH_MAX    63
			#define DEFAULT_UPDATE_URL  "http://www.hytem.net"
		char update_url[UPDATE_URL_LENGTH_MAX + 1];         ///< URL to check for firmware update.

		// 124
		int32_t multiusers_option;
			#define DEFAULT_MULTIUSERS_OPTION   1

		// Parameters depending on platform type.

		// 128
			#if defined(_BANANA_PRO)

				// Min on connector C6: PB3
				#define MINIMAL_GPIO  35
				// Max on connector C6: PI21
				#define MAXIMAL_GPIO  277
				#define GPIO_CONTROLLER "/dev/gpiochip0"

			#elif defined(_BANANA_PI_M1)

				// Min on connector C6: PB20
				#define MINIMAL_GPIO  52
				// Max on connector C6: PI19
				#define MAXIMAL_GPIO  275
				#define GPIO_CONTROLLER "/dev/gpiochip0"

			#elif defined(_RASPBERRY_PI)

				// Min on GPIO port: Pin #3
				#define MINIMAL_GPIO  2
				// Max on GPIO port: Pin #13
				#define MAXIMAL_GPIO  27
				#define GPIO_CONTROLLER "/dev/gpiochip0"

			#elif defined(_PHYBOARD)

				#define MINIMAL_GPIO  1
				#define MAXIMAL_GPIO  300

			#else
				#warning "Missing platform board name - Low-level GPIO support not available"
			#endif

			#define NORMAL_POLARITY     0
			#define INVERTED_POLARITY   1

		int32_t local_remote_gpio;
			#if defined(_BANANA_PRO)
				// Pin 12
				#define DEFAULT_LOCAL_REMOTE_GPIO      259
			#elif defined(_BANANA_PI_M1)
				// Pin 12
				#define DEFAULT_LOCAL_REMOTE_GPIO      226
			#elif defined(_RASPBERRY_PI)
				// Pin 12
				#define DEFAULT_LOCAL_REMOTE_GPIO      18
			#elif defined(_PHYBOARD)
				// Pin 28 - GPIO 197
				#define DEFAULT_LOCAL_REMOTE_GPIO      197
			#endif

		// 132
		int32_t local_remote_polarity;
			#define DEFAULT_LOCAL_REMOTE_POLARITY  0

		// 136
		int32_t user_admin_gpio;
			#if defined(_BANANA_PRO)
				// Pin 16
				#define DEFAULT_USER_ADMIN_GPIO        244
			#elif defined(_BANANA_PI_M1)
				// Pin 16
				#define DEFAULT_USER_ADMIN_GPIO        244
			#elif defined(_RASPBERRY_PI)
				// Pin 16
				#define DEFAULT_USER_ADMIN_GPIO        23
			#elif defined(_PHYBOARD)
				// Pin 30 - GPIO 198
				#define DEFAULT_USER_ADMIN_GPIO        198
			#endif

		// 140
		int32_t user_admin_polarity;
			#define DEFAULT_USER_ADMIN_POLARITY  0

		// 144
			#define SYNC_OUTPUT_TYPE_GPIO        0
			#define SYNC_OUTPUT_TYPE_CONTROLLER  1
		int32_t sync_output_type;
			#if defined(_BANANA_PRO)
				#define DEFAULT_SYNC_OUTPUT_TYPE   SYNC_OUTPUT_TYPE_GPIO
			#elif defined(_BANANA_PI_M1)
				#define DEFAULT_SYNC_OUTPUT_TYPE   SYNC_OUTPUT_TYPE_GPIO
			#elif defined(_RASPBERRY_PI)
				#define DEFAULT_SYNC_OUTPUT_TYPE   SYNC_OUTPUT_TYPE_CONTROLLER
			#elif defined(_PHYBOARD)
				#define DEFAULT_SYNC_OUTPUT_TYPE   SYNC_OUTPUT_TYPE_CONTROLLER
			#endif

		// 148
		int32_t sync_output_clock_gpio;
			#if defined(_BANANA_PRO)
				// Pin 31
				#define DEFAULT_SYNC_OUTPUT_CLOCK_GPIO   277
			#elif defined(_BANANA_PI_M1)
				// Pin 21
				#define DEFAULT_SYNC_OUTPUT_CLOCK_GPIO   269
			#elif defined(_RASPBERRY_PI)
				// Pin 31
				#define DEFAULT_SYNC_OUTPUT_CLOCK_GPIO     6
			#else
				#define DEFAULT_SYNC_OUTPUT_CLOCK_GPIO     0
			#endif

		// 152
		int32_t sync_output_data_gpio;
			#if defined(_BANANA_PRO)
				// Pin 33
				#define DEFAULT_SYNC_OUTPUT_DATA_GPIO     45
			#elif defined(_BANANA_PI_M1)
				// Pin 23
				#define DEFAULT_SYNC_OUTPUT_DATA_GPIO     267
			#elif defined(_RASPBERRY_PI)
				// Pin 33
				#define DEFAULT_SYNC_OUTPUT_DATA_GPIO     13
			#else
				#define DEFAULT_SYNC_OUTPUT_DATA_GPIO     0
			#endif

		// 156
		int32_t sync_output_enable_gpio;
			#if defined(_BANANA_PRO)
				// Pin 29
				#define DEFAULT_SYNC_OUTPUT_ENABLE_GPIO   35
			#elif defined(_BANANA_PI_M1)
				// Pin 19
				#define DEFAULT_SYNC_OUTPUT_ENABLE_GPIO   268
			#elif defined(_RASPBERRY_PI)
				// Pin 29
				#define DEFAULT_SYNC_OUTPUT_ENABLE_GPIO   5
			#elif defined(_PHYBOARD)
				// Pin 36
				#define DEFAULT_SYNC_OUTPUT_ENABLE_GPIO    192
			#else
				#define DEFAULT_SYNC_OUTPUT_ENABLE_GPIO     0
			#endif

		// 160
		int32_t sync_output_speed;
			#define MINIMAL_SYNC_OUTPUT_SPEED   100
			#define DEFAULT_SYNC_OUTPUT_SPEED   100000
			#define MAXIMAL_SYNC_OUTPUT_SPEED   125000000

		// 164
		int32_t sync_output_devices_order;
			#define SYNC_OUTPUT_LOWEST_FIRST    0
			#define SYNC_OUTPUT_HIGHEST_FIRST  1
			#define DEFAULT_SYNC_OUTPUT_DEVICES_ORDER SYNC_OUTPUT_LOWEST_FIRST

		// 168
		int32_t sync_output_width;
			#define SYNC_OUTPUT_8_BITS     0
			#define SYNC_OUTPUT_16_BITS    1
			#define DEFAULT_SYNC_OUTPUT_WIDTH  SYNC_OUTPUT_8_BITS

		// 172
		int32_t sync_output_data_params;
			#define SYNC_OUTPUT_LSB_FIRST    0
			#define SYNC_OUTPUT_MSB_FIRST    1
			#define DEFAULT_SYNC_OUTPUT_BYTES_ORDER  SYNC_OUTPUT_LSB_FIRST
			#define SYNC_OUTPUT_BYTES_ORDER_MASK (SYNC_OUTPUT_LSB_FIRST | SYNC_OUTPUT_MSB_FIRST)

			#define SYNC_OUTPUT_NORMAL_DATA    0
			#define SYNC_OUTPUT_INVERTED_DATA  2
			#define DEFAULT_SYNC_OUTPUT_DATA_POLARITY  SYNC_OUTPUT_NORMAL_DATA
			#define SYNC_OUTPUT_DATA_POLARITY_MASK (SYNC_OUTPUT_NORMAL_DATA | SYNC_OUTPUT_INVERTED_DATA)

		// Network parameters

			#define LAST_REGISTRABLE_TCP_PORT   49151
		// 176
		int32_t external_api_server_base_port;
			#define DEFAULT_EXTERNAL_API_SERVER_BASE_PORT   10001

		// 180
		int32_t internal_api_server_port;
			#define DEFAULT_INTERNAL_SERVER_PORT   4000

		// Current GPIO status

		// 184
			#define NUMBER_OF_GPIOS             8
		uint8_t gpio_status[NUMBER_OF_GPIOS];
			#define LOCAL_REMOTE_GPIO_INDEX       0
			#define USER_ADMIN_GPIO_INDEX         1

		// 192
		uint8_t ipv4_address[4]; ///< Configured IPv4 address (may not be the current address if DHCP is used)

		// 196
		uint8_t ipv4_netmask[4]; ///< Configured IPv4 network mask.

		// 200
		uint8_t ipv4_gateway[4]; ///< Configured IPv4 address of the default route gateway.

		// 204
		uint8_t ipv4_nameserver[4]; ///< Configured IPv4 address of the nameserver.

		// 208
		uint8_t ipv4_dhcp; ///< Dynamic I4 configuration using DHCP.
			#define DEFAULT_IPV4_DHCP       1

		// 209
		uint8_t use_https; ///< Use HTTPS for web HMI
			#define DEFAULT_USE_HTTPS       0

		// 210
		uint8_t board_model;
			#define BOARD_MODEL_UNKNOWN           0
			#define BOARD_MODEL_BANANA_PRO        1
			#define BOARD_MODEL_BANANA_PI_M1      2
			#define BOARD_MODEL_BANANA_PI_M1P     3
			#define BOARD_MODEL_PHYBOARD          4
			#define BOARD_MODEL_RASPBERRY_PI_3    5

		// 211
		uint8_t dhcp_server; ///< Run a DHCP server
			#define DEFAULT_DHCP_SERVER       0

		// 212
			#define SYSTEM_NAME_LENGTH_MAX    32
			#define DEFAULT_SYSTEM_NAME       "Hytem System"
		char system_name[SYSTEM_NAME_LENGTH_MAX + 1];

		// 247
		uint8_t padding[512 - 247 - sizeof(pthread_mutex_t)];  ///< Padding up to 256 bytes

		// ???
		pthread_mutex_t mutex; ///< Process shared mutex to access the attenuators values.

	// 512

	} config_global_parameters_t;

	extern config_global_parameters_t *Config_global_parameters;

	#define CONFIG_GLOBAL_PARAMETERS_SIZE  sizeof(config_global_parameters_t)


	// Accessors

	#define Config_system_version              Config_global_parameters->version
	#define Config_supported_devices           Config_global_parameters->supported_devices
	#define Config_number_of_groups            Config_global_parameters->number_of_groups
	#define Config_devices_per_group           Config_global_parameters->devices_per_group
	#define Config_number_of_devices           Config_global_parameters->number_of_devices
	#define Config_attenuators_range           Config_global_parameters->attenuators_range
	#define Config_throws_per_switch           Config_global_parameters->throws_per_switch
	#define Config_product_number              Config_global_parameters->product_number
	#define Config_serial_number               Config_global_parameters->serial_number
	#define Config_update_url                  Config_global_parameters->update_url
	#define Config_multiusers_option           Config_global_parameters->multiusers_option
	#define Config_ipv4_address                Config_global_parameters->ipv4_address
	#define Config_ipv4_netmask                Config_global_parameters->ipv4_netmask
	#define Config_ipv4_gateway                Config_global_parameters->ipv4_gateway
	#define Config_ipv4_nameserver             Config_global_parameters->ipv4_nameserver
	#define Config_ipv4_dhcp                   Config_global_parameters->ipv4_dhcp
	#define Config_use_https                   Config_global_parameters->use_https
	#define Config_dhcp_server                 Config_global_parameters->dhcp_server
	#define Config_board_model                 Config_global_parameters->board_model
	#define Config_system_name                 Config_global_parameters->system_name

	int Config_set_supported_devices(int value);
	int Config_set_number_of_groups(int value);
	int Config_set_devices_per_group(int value);
	int Config_set_attenuators_range(int value);
	int Config_set_throws_per_switch(int value);
	int Config_set_product_number(const char *string);
	int Config_set_serial_number(const char *string);
	int Config_set_update_url(const char *string);
	int Config_set_multiusers_option(int value);
	int Config_set_sync_output_type(int value);
	int Config_set_sync_output_speed(int value);
	int Config_set_sync_output_devices_order(int value);
	int Config_set_sync_output_width(int value);
	int Config_set_sync_output_data_params(int value);
	int Config_set_ipv4_address(const char *address);
	int Config_set_ipv4_netmask(const char *netmask);
	int Config_set_ipv4_gateway(const char *gateway);
	int Config_set_ipv4_nameserver(const char *nameserver);
	int Config_set_ipv4_dhcp(int value);
	int Config_set_use_https(int value);
	int Config_set_dhcp_server(int value);
	int Config_set_board_model(int value);
	int Config_set_system_name(const char *string);

	#define Config_local_remote_gpio            Config_global_parameters->local_remote_gpio
	#define Config_local_remote_polarity        Config_global_parameters->local_remote_polarity
	#define Config_local_remote_switch_on_local Config_global_parameters->gpio_status[LOCAL_REMOTE_GPIO_INDEX]

	#define Config_user_admin_gpio              Config_global_parameters->user_admin_gpio
	#define Config_user_admin_polarity          Config_global_parameters->user_admin_polarity
	#define Config_user_admin_switch_on_user    Config_global_parameters->gpio_status[USER_ADMIN_GPIO_INDEX]

	#define Config_sync_output_type             Config_global_parameters->sync_output_type
	#define Config_sync_output_clock_gpio       Config_global_parameters->sync_output_clock_gpio
	#define Config_sync_output_data_gpio        Config_global_parameters->sync_output_data_gpio
	#define Config_sync_output_enable_gpio      Config_global_parameters->sync_output_enable_gpio
	#define Config_sync_output_speed            Config_global_parameters->sync_output_speed
	#define Config_sync_output_devices_order    Config_global_parameters->sync_output_devices_order
	#define Config_sync_output_width            Config_global_parameters->sync_output_width
	#define Config_sync_output_data_params      Config_global_parameters->sync_output_data_params

	#define Config_external_api_server_base_port Config_global_parameters->external_api_server_base_port
	#define Config_internal_api_server_port      Config_global_parameters->internal_api_server_port


	#define Config_parameters_mutex          Config_global_parameters->mutex


	void  Config_no_save (void);
	void  Config_save    (void);


	/// DEVICES SPECIFIC PARAMETERS TABLE

	typedef struct {
		// 0
		#define DEVICE_NAME_LENGTH_MAX   7
		char name[DEVICE_NAME_LENGTH_MAX + 1];  ///< Device name.

		// 8
		#define DEVICE_ID_LENGTH_MAX  7
		char id[DEVICE_ID_LENGTH_MAX + 1];      ///< Device id.

		// 16
		#define DEVICE_TYPE_EMPTY          0
		#define DEVICE_TYPE_ATTENUATOR     1
		#define DEVICE_TYPE_SWITCH         2
		#define DEVICE_TYPE_SPST           4

		#define MINIMAL_DEVICE_TYPE    DEVICE_TYPE_EMPTY
		#define MAXIMAL_DEVICE_TYPE    DEVICE_TYPE_SPST
		#define DEFAULT_DEVICE_TYPE    DEVICE_TYPE_ATTENUATOR
		uint32_t type; ///< Type of device.

		// 20
		int32_t current_value;  ///< Current value of the device.

		// 24
		#define DEVICE_LOCK_ID_FREE 0
		#define DEVICE_LOCK_ID_MAX_VALUE   65535

		uint32_t lock_id; ///< Lock Id:  0 Free, or session ID (uint16 value).

		// 28
		#define TIMEOUT_COUNTER_FOR_UNLOCK   20  // Seconds.
		#define SHORT_TIMEOUT_FOR_UNLOCK     10  // Seconds.
		int32_t counter; ///< Timeout counter for forced unlocking

		// 32
		int32_t min_value;
		// 36
		int32_t max_value;
		// 40
		int32_t default_value;
		// 44
		int32_t granularity;
		// 48
		float   display_factor;
		// 48 + sizeof(float)
		uint8_t lock_ip[4]; ///< IP address of the locking client
		// 52 + sizeof(float)
		uint8_t padding[64-52-sizeof(float)]; ///< Padding up to 64 bytes.

	} config_device_t;

	#define CONFIG_DEVICE_SIZE  sizeof(config_device_t)

	#define CONFIG_DEVICE_TABLE_SIZE (CONFIG_DEVICE_SIZE * MAXIMAL_NUMBER_OF_DEVICES)

	extern config_device_t *Config_device_table;

	// Accessors

	int   Config_get_device_type(int dev);
	int   Config_set_device_type(int dev, int type);

	char *Config_get_device_name(int dev);
	int   Config_set_device_name(int dev, const char *name);

	char *Config_get_device_id(int dev);
	int   Config_set_device_id(int dev, const char *id);

	int   Config_get_device_lock_ip(int dev, int byte);
	int   Config_set_device_lock_ip(int dev, int byte, int value);


	// GROUPS SPECIFIC PARAMETERS TABLE

	typedef struct {
		// 0
		#define GROUP_NAME_LENGTH_MAX 7
		char name[GROUP_NAME_LENGTH_MAX + 1]; ///< Group name

		// 8
		uint8_t padding[8]; ///< Padding up to 16 bytes.

	} config_group_t;


	#define CONFIG_GROUP_SIZE  sizeof(config_group_t)

	#define CONFIG_GROUPS_TABLE_SIZE (CONFIG_GROUP_SIZE * MAXIMAL_NUMBER_OF_GROUPS)

	extern config_group_t *Config_group_table;


	#define SHARED_MEMORY_SIZE (CONFIG_GLOBAL_PARAMETERS_SIZE + CONFIG_DEVICE_TABLE_SIZE + CONFIG_GROUPS_TABLE_SIZE)


	char *Config_get_group_name(int grp);
	int   Config_set_group_name(int grp, const char *name);



	// FILENAMES CONFIGURATION

	extern char Config_system_version_file[];

		#define SYSTEM_VERSION_FILE_MAX_LENGTH    63
		#define DEFAULT_SYSTEM_VERSION_FILE   "/etc/hytem-version"

	extern char Config_users_database_file[];

		#define USERS_DATABASE_FILE_MAX_LENGTH    63
		#define DEFAULT_USERS_DATABASE_FILE   "/data/config/users.tbl"


	#define DEVICES_SHM_FILE   "/hytem-system.shm"


	// LOG PARAMETERS

	extern char Config_logger_error_output[];
		#define LOGGER_ERROR_OUTPUT_MAX_LENGTH    63
		#define DEFAULT_LOGGER_ERROR_OUTPUT   "/var/log/devices-server.err"

	extern char Config_logger_messages_output[];
		#define LOGGER_MESSAGES_OUTPUT_MAX_LENGTH    63
		#define DEFAULT_LOGGER_MESSAGES_OUTPUT   "/var/log/devices-server.msg"

	extern char Config_logger_debug_output[];
		#define LOGGER_DEBUG_OUTPUT_MAX_LENGTH    63
		#define DEFAULT_LOGGER_DEBUG_OUTPUT   "/dev/null"

	extern int Config_logger_debug_level;
		#define DEFAULT_LOGGER_DEBUG_LEVEL   30

	#define	LOCAL_REMOTE_SWITCH_FORCE_LOCAL_PATHNAME   "/tmp/force-local"
	#define LOCAL_REMOTE_SWITCH_FORCE_REMOTE_PATHNAME  "/tmp/force-remote"

	#define	USER_ADMIN_FORCE_USER_PATHNAME             "/tmp/force-user"
	#define USER_ADMIN_FORCE_ADMIN_PATHNAME            "/tmp/force-admin"


#endif
