//  -------------------------------------------------------------------------
/// \brief Attenuator player implementation.
///
/// \author Logilin 2016-2018 - http://www.logilin.fr
///

const PLAYER_MASTER_MODE = 0;
const PLAYER_SLAVE_MODE = 1;


/// \brief  Get and display the names of devices and groups.
///
/// Get the names of devices and groups from server shared memory,
/// store them into the window instance, then call \ref update_player_names().
///
/// This the entry point for the initial display of the window.
///
/// \called-by  player.php:load_user_permissions_and_display_players()
///
function load_and_display_all_names_and_types()
{
	var values;
	var i;
	var xhr = get_xhr_object();

	if (xhr != null) {

		xhr.onreadystatechange = function()
		{
			if ((xhr.readyState == 4) && (xhr.status == 200)) {
				values = xhr.responseText.split("%");
				for (i = 0; i  < window.nb_players * window.nb_groups; i ++) {
					window.dev_names[i] = values[32 + i * 12 + 0];
					window.dev_types[i] = values[32 + i * 12 + 1];
				}
				for (i = 0; i < window.nb_groups; i ++) {
					window.grp_names[i] = values[32 + window.nb_players * window.nb_groups * 12 + i];
					document.getElementById('group-name-' + i).innerHTML = window.grp_names[i];
				}
				load_sequences_list();
			}
		};

		xhr.open("GET", "cgi-bin/get-shm-data", true);
		xhr.send();
	}
}



/// \brief Start the loading of sequences by the attenuators ones.
///
/// \called-by  load_and_display_all_names_and_types()
///
function load_sequences_list()
{
	load_attenuator_sequences_list_then_others();
}



var _Attenuator_sequences_list = [];

/// \brief  Fill the _Attenuator_sequences_list array, then load other sequences.
///
/// \called-by  load_sequences_list()
///
function load_attenuator_sequences_list_then_others()
{
	var xhr = get_xhr_object();

	_Attenuator_sequences_list = [];

	if (xhr != null) {
		xhr.onreadystatechange = function()
		{
			if ((xhr.readyState == 4) && (xhr.status == 200)) {
				xhr.responseText.split("%").forEach(function(item, index) {
					if (item != "")
						_Attenuator_sequences_list.push(item);
				});
				load_switch_sequences_list_then_others();
			}
		};
		xhr.open("GET", "cgi-bin/list-sequences?type=1", true);
		xhr.send();
	}
}



var _Switch_sequences_list = [];

/// \brief  Fill the _Switch_sequences_list array then load the other sequences.
///
/// \called-by  load_attenuator_sequences_list_then_others()
///
function load_switch_sequences_list_then_others()
{
	var xhr = get_xhr_object();

	_Switch_sequences_list = [];

	if (xhr != null) {
		xhr.onreadystatechange = function()
		{
			if ((xhr.readyState == 4) && (xhr.status == 200)) {
				xhr.responseText.split("%").forEach(function(item) {
					if (item != "")
						_Switch_sequences_list.push(item);
				});
				load_spst_sequences_list_then_update_display();
			}
		};
		xhr.open("GET", "cgi-bin/list-sequences?type=2", true);
		xhr.send();
	}
}



var _Spst_sequences_list = [];

/// \brief  Fill the _Spst_sequences_list array.
///
/// \called-by  load_switch_sequences_list_then_others()
///
function load_spst_sequences_list_then_update_display()
{
	var xhr = get_xhr_object();

	_Spst_sequences_list = [];

	if (xhr != null) {
		xhr.onreadystatechange = function()
		{
			if ((xhr.readyState == 4) && (xhr.status == 200)) {
				xhr.responseText.split("%").forEach(function(item) {
					if (item != "")
						_Spst_sequences_list.push(item);
				});
				update_player_names_and_sequences();
				load_and_display_player(-1, false);
			}
		};
		xhr.open("GET", "cgi-bin/list-sequences?type=4", true);
		xhr.send();
	}
}



/// \brief Display the devices names for all current players.
///
/// The names are read from the window instance tables.
///
/// \called-by - load_spst_sequences_list_then_update_display()
///              player.php:group_name_cb()
///
function update_player_names_and_sequences()
{
	for (var i = 0; i  < window.nb_players; i ++) {
		document.getElementById('player-dev-name-' + i).innerHTML = window.dev_names[i + window.current_group * window.nb_players];
	}
}



/// \brief Load and display all players.
///
/// \called-by - player.php:group_name_cb()
///
function display_all_players()
{
	for (var i = 0; i < window.nb_players; i ++)
		display_player(i);
}



/// \brief Load the data for all players, display one or all of them.
///
/// \param player   The player index (or -1 for all players).
/// \param slave    Boolean indicating if the slaves must be updated too.
///
/// \called-by - load_spst_sequences_list_then_update_display()
///            - set_synchro()
///              sequences_selection_changed_cb()
///              start_player()
///              pause_player()
///              stop_player()
///
function load_and_display_player(player, update_slaves)
{
	var xhr = get_xhr_object();

	if (xhr != null) {
		xhr.onreadystatechange = function()
		{
			if ((xhr.readyState == 4) && (xhr.status == 200)) {

				var statuses = xhr.responseText.split(",");

				for (var i = 0; i  < window.nb_players * window.nb_groups; i ++) {
					window.synchro_mode[i]    = (statuses[i].substring(0,1)  == "M") ? PLAYER_MASTER_MODE : PLAYER_SLAVE_MODE;
					window.synchro_action[i]  = statuses[i].substring(1,2);            // P(lay), W(ait), N(one).
					window.synchro_master[i]  = parseInt(statuses[i].substring(2,5));  // IP or -1.
					window.synchro_channel[i] = parseInt(statuses[i].substring(6,9));  // Device number or -1.
					if (window.synchro_channel[i] > 0)
						window.synchro_channel[i]--;
					window.synchro_file[i]    = statuses[i].substring(10);             // Sequence or 'None'.
				}

				if (player != -1) {

					display_player(player);
					if (update_slaves) {
						var dev = window.current_group * window.nb_players + player;

						for (var i = 0; i < window.nb_players; i++) {
							var slave = window.current_group * window.nb_players + i;

							if ((i != player)
							 && (window.synchro_mode[slave] == PLAYER_SLAVE_MODE)
							 && (window.synchro_master[slave] == -1)
							 && (window.synchro_channel[slave] == dev))
								display_player(i);
						}
					}

				} else {

					for (var i = 0; i < window.nb_players; i ++) {
						fill_sequences_list(i);
						display_player(i);
					}
				}
			}
		};
		xhr.open("GET", "action/get-device-statuses.php", true);
		xhr.send();
	}
}



/// \brief Display a player (data have previously been loaded).
///
/// \param  player - The player index.
///
/// \called-by   display_all_players()
///              load_and_display_player()
///
function display_player(player)
{
	var dev = player + window.current_group * window.nb_players;

	var synchro_mode_select   = document.getElementById('synchro-mode-'     + player);
	var master_host_select    = document.getElementById('master-host-'      + player);
	var master_group_select   = document.getElementById('master-group-'     + player);
	var master_channel_select = document.getElementById('master-channel-'   + player);
	var sequence_select       = document.getElementById('sequence-'         + player);
	var player_buttons        = document.getElementById('player-buttons-'   + player);
	var start_button          = document.getElementById('start-button-img-' + player);
	var pause_button          = document.getElementById('pause-button-img-' + player);
	var stop_button           = document.getElementById('stop-button-img-'  + player);
	var player_name           = document.getElementById('player-dev-name-'  + player);

	if (User_permissions[dev] == 0) {
		player_name.style = "color: #808080;";

		synchro_mode_select.value = PLAYER_MASTER_MODE;
		synchro_mode_select.setAttribute("disabled", true);
		synchro_mode_select.style = "background: #A0A0A0;";

		master_host_select.options.length = 0;
		master_host_select.disabled = true;
		master_host_select.style = "background: #A0A0A0;";

		master_group_select.options.length = 0;
		master_group_select.disabled = true;
		master_group_select.style = "background: #A0A0A0;";

		master_channel_select.options.length = 0;
		master_channel_select.disabled = true;
		master_channel_select.style = "background: #A0A0A0;";

		sequence_select.disabled = "disabled";
		sequence_select.style = "background: #A0A0A0;";

		player_buttons.disabled = true;

		start_button.disabled = true;
		start_button.src = 'png/bt-play-32-disabled.png';

		stop_button.disabled = true;
		stop_button.src = 'png/bt-stop-32-disabled.png';

		pause_button.disabled = true;
		pause_button.src = 'png/bt-pause-32-disabled.png';
		return;
	}

	player_name.style = "";

	synchro_mode_select.disabled = false;
	synchro_mode_select.style = "";

	master_host_select.disabled =  false
	master_host_select.style = "";

	master_group_select.disabled =false
	master_group_select.style = "";

	master_channel_select.disabled = false
	master_channel_select.style = "";

	sequence_select.disabled = false;
	sequence_select.style = "";

	player_buttons.disabled = false
	start_button.disabled = false
	pause_button.disabled = false
	stop_button.disabled = false

	if (window.synchro_mode[dev] == PLAYER_MASTER_MODE) {

		player_buttons.hidden = false;

		synchro_mode_select.value = PLAYER_MASTER_MODE;

		master_host_select.options.length = 0;
		master_host_select.disabled = "disabled";

		master_group_select.options.length = 0;
		master_group_select.disabled = "disabled";

		master_channel_select.options.length = 0;
		master_channel_select.disabled = "disabled";

		if (window.synchro_file[dev] == 'None') {

			start_button.src = 'png/bt-play-32-disabled.png';
			start_button.disabled = "disabled";

			stop_button.src = 'png/bt-stop-32-disabled.png';
			stop_button.disabled = "disabled";

			pause_button.src = 'png/bt-pause-32-disabled.png';
			pause_button.disabled = "disabled";

		} else {

			if ((synchro_action[dev] == 'W') || (synchro_action[dev] == 'N')) {
				start_button.src = 'png/bt-play-32.png';
				start_button.disabled = false;
			} else {
				start_button.src = 'png/bt-play-32-disabled.png';
				start_button.disabled = "disabled";
			}
			if ((synchro_action[dev] == 'P') || (synchro_action[dev] == "W")) {
				stop_button.src = 'png/bt-stop-32.png';
				stop_button.disabled = false;
			} else {
				stop_button.src = 'png/bt-stop-32-disabled.png';
				stop_button.disabled = "disabled";
			}
			if (synchro_action[dev] == "P") {
				pause_button.src = 'png/bt-pause-32.png';
				pause_button.disabled = false;
			} else {
				pause_button.src = 'png/bt-pause-32-disabled.png';
				pause_button.disabled = "disabled";
			}
		}

	} else {

		synchro_mode_select.value = PLAYER_SLAVE_MODE;

		fill_hosts_list(player, false);

		master_host_select.disabled    = "";
		master_group_select.disabled   = "";
		master_channel_select.disabled = "";

		player_buttons.hidden = true;
	}

	if ((synchro_action[dev] == "W") || (synchro_action[dev] == "P")) {

		synchro_mode_select.disabled   = "disabled";
		master_host_select.disabled    = "disabled";
		master_group_select.disabled   = "disabled";
		master_channel_select.disabled = "disabled";
		sequence_select.disabled       = "disabled";

	} else {
		synchro_mode_select.disabled   = "";
		sequence_select.disabled       = "";
	}

	fill_sequences_list(player);
/*	sequence_select.selectedIndex = 0;
	for (var i = 0; i < sequence_select.options.length; i ++) {
		if (sequence_select.options[i].value == synchro_file[dev]) {
			sequence_select.selectedIndex = i;
			break;
		}
	}
*/
}



/// \brief Fill the host list with all host IP addresses seen so far.
///
/// \param player   - The player index (starting from zero).
/// \param notify   - Do we need to call the hosts_selection_changed_cb() callback?
///
/// \called-by   fill_hosts_list()
///              synchros_selection_changed_cb()
///
function fill_hosts_list(player, notify)
{
	var master_group_select = document.getElementById('master-group-' + player);
	master_group_select.selectedIndex  = -1;
	master_group_select.options.length =  0;

	var xhr = get_xhr_object();
	if (xhr != null) {

		xhr.onreadystatechange = function()
		{
			if ((xhr.readyState == 4) && (xhr.status == 200)) {

				var master_host_select  = document.getElementById('master-host-' + player);
				master_host_select.options.length =  0;

				if (window.nb_players * window.nb_groups > 1) {
					master_host_select.options[0] = new Option("Local", -1, false, false);
					master_host_select.options[0].nb_groups   = window.nb_groups;
					master_host_select.options[0].nb_channels = window.nb_players * window.nb_groups;
				}

				// Walk toward the hosts list.
				var number_of_hosts = parseInt(xhr.responseText.substring(0,3));
				for (var i = 0; i < number_of_hosts; i ++) {
					var host_addr   = parseInt(xhr.responseText.substring(4+i*12+0, 4+i*12+3));
					var nb_groups   = parseInt(xhr.responseText.substring(4+i*12+4, 4+i*12+7));
					var nb_channels = parseInt(xhr.responseText.substring(4+i*12+8, 4+i*12+11));
					// Refuse host with incorrect IP, number of groups or channels.
					if ((host_addr < 0) || (nb_groups < 1) || (nb_channels < 1))
						continue;
					var opt  = new Option(host_addr, host_addr, false, false);
					opt.nb_groups   = nb_groups;
					opt.nb_channels = nb_channels;
					master_host_select.options.add(opt);

					if (host_addr == window.synchro_master[player + window.current_group * window.nb_players])
						master_host_select.selectedIndex = master_host_select.options.length - 1;
				}

				if ((master_host_select.selectedIndex == -1) && (master_host_select.options.length > 0))
					master_host_select.selectedIndex = 0;

				if (notify)
					hosts_selection_changed_cb(player);
				else
					fill_groups_list(player, false);
			}
		};

		xhr.open("GET", "action/get-unit-statuses.php");
		xhr.send();
	}
}



/// \brief Fill the groups menu.
///
/// \param player   - The player index
/// \param notify   - Do we need to call the groups_selection_changed_cb() callback?
///
/// \called-by   fill_hosts_list()
///              hosts_selection_changed_cb()
///
/// \returns   true is a channel is selectable, false otherwise.
///
function fill_groups_list(player, notify)
{
	var master_group_select = document.getElementById('master-group-' + player);
	master_group_select.selectedIndex = -1;
	master_group_select.options.length = 0;

	var master_host_select  = document.getElementById('master-host-' + player);
	var nb_grp = master_host_select.options[master_host_select.selectedIndex].nb_groups
	var nb_chn = master_host_select.options[master_host_select.selectedIndex].nb_channels;

	// Get the currently selected channel.
	var selected_channel = window.synchro_channel[player + window.nb_players * window.current_group];
	if ((selected_channel > nb_chn) || (selected_channel < 0))
		selected_channel = 0;


	var grp = Math.trunc(selected_channel * nb_grp / nb_chn);
	var opt_index = 0;
	for (var i = 0; i < nb_grp; i ++) {
		if (is_group_selectable_as_master(player, i, Math.trunc(nb_chn / nb_grp))) {
			master_group_select.options[opt_index] = new Option(i+1, i, false, false);
			master_group_select.options[opt_index].first_channel = Math.trunc(i * nb_chn / nb_grp);
			if (grp == i)
				master_group_select.selectedIndex = opt_index;
			opt_index ++;
		}
	}

	if ((master_group_select.selectedIndex == -1) && (master_group_select.options.length > 0))
		master_group_select.selectedIndex = 0;

	fill_channels_list(player, notify)

	return true;
}



function is_group_selectable_as_master(player, group)
{
	if (document.getElementById('master-host-' + player).selectedIndex != 0)
		// Remote synchronisation.
		return true;

	if (window.nb_players * window.nb_groups == 1)
		// Remote synchronisation
		return true;

	var device = player + window.nb_players * window.current_group;

	var first = group * window.nb_players;

	for (var i = 0; i < window.nb_players; i ++) {

		var index = first + i;

		if (! is_local_channel_selectable_as_master(device, index))
			continue;

		return true;
	}
	return false;
}




/// \brief Fill the channels menu.
///
/// \param player   The player index (starting from zero).
/// \param notify - Do we need to call the channels_selection_changed_cb() callback?
///
/// \returns  false if channel list is empty, true otherwise.
///
/// \called-by   fill_groups_list()
///              groups_selection_changed_cb()
///
function fill_channels_list(player, notify)
{
	var dev = player + window.current_group * window.nb_players;

	var master_host_select    = document.getElementById('master-host-' + player);
	var master_group_select   = document.getElementById('master-group-' + player);
	var master_channel_select = document.getElementById('master-channel-' + player);

	var nb_grp = master_host_select.options[master_host_select.selectedIndex].nb_groups;
	var nb_chn = master_host_select.options[master_host_select.selectedIndex].nb_channels;


	if (master_group_select.selectedIndex < 0)
		return false;

	var cur_grp = master_group_select.selectedOptions[0].value;

	var sel_chn = window.synchro_channel[dev];
	var first = Math.trunc(cur_grp * nb_chn / nb_grp);

	master_channel_select.selectedIndex = -1;
	master_channel_select.options.length =  0;

	for (var i = 0; i < Math.trunc(nb_chn/nb_grp); i ++) {
		var index = first + i;

		if ((master_host_select.selectedIndex == 0) && (window.nb_players * window.nb_groups > 1)) {
			// Local synchronisation.


			if (! is_local_channel_selectable_as_master(dev, index))
				continue;

			master_channel_select.add(new Option(window.dev_names[index], index, false, false));
		} else {
			// Remote.
			master_channel_select.add(new Option(""+(i+1)+" ("+(index+1)+")", index, false, false));
		}
		if (index == sel_chn)
			master_channel_select.selectedIndex = master_channel_select.options.length - 1;
	}
	if (master_channel_select.options.length == 0)
		return false;

	if (notify)
		channels_selection_changed_cb(player);
	return true;
}



function is_local_channel_selectable_as_master(player, channel)
{
	if (channel == player)
		return false;

	if (window.synchro_mode[channel] != PLAYER_MASTER_MODE)
		return false;

	if (User_permissions[channel] == 0)
		return false;

	return true;
}



/// \called-by  load_and_display_player()
///             display_player()
///
function fill_sequences_list(player)
{
	var sequences_select = document.getElementById('sequence-' + player);
	sequences_select.selectedIndex = -1;
	sequences_select.options.length = 0;

	var device = player + window.current_group * window.nb_players;

	if (localStorage.getItem('HytemLang') == "fr")
		sequences_select.options[0] = new Option("Aucune", "Aucune", false, false);
	else
		sequences_select.options[0] = new Option("None", "None", false, false);
	sequences_select.selectedIndex = 0;

	var sequences_list;

	if (window.dev_types[device] == 1) {
		sequences_list = _Attenuator_sequences_list;
	} else if (window.dev_types[device] == 2) {
		sequences_list = _Switch_sequences_list;
	} else {
		sequences_list = _Spst_sequences_list;
	}

	sequences_list.forEach(function(item, index) {
		sequences_select.options[index + 1] = new Option(item, item, false, false);
		if (item == window.synchro_file[device])
			sequences_select.selectedIndex = index + 1;
	});
}



/// \brief Callback for changes in synchronisation type menu.
///
/// \param player - The player index.
///
/// \¢alled-by  player.php "onchange" event.
///
function synchros_selection_changed_cb(player)
{
	var synchro_mode_select   = document.getElementById('synchro-mode-'   + player);
	var master_host_select    = document.getElementById('master-host-'    + player);
	var master_group_select   = document.getElementById('master-group-'   + player);
	var master_channel_select = document.getElementById('master-channel-' + player);
	var player_buttons        = document.getElementById('player-buttons-' + player);

	if (synchro_mode_select.selectedIndex == PLAYER_MASTER_MODE) {

		master_host_select.disabled    = "disabled";
		master_group_select.disabled   = "disabled";
		master_channel_select.disabled = "disabled";

		master_host_select.options.length    = 0;
		master_group_select.options.length   = 0;
		master_channel_select.options.length = 0;

		player_buttons.hidden = false;

		set_synchro(player, PLAYER_MASTER_MODE, -1, -1);

	} else {

		master_host_select.disabled    = "";
		master_group_select.disabled   = "";
		master_channel_select.disabled = "";

		player_buttons.hidden = true;

		window.synchro_master[player + window.nb_players * window.current_group] = -1;

		fill_hosts_list(player, true);
	}
}



/// \brief   Notify the player module of its synchronisation.
///          Then refresh the concerned player.
///
/// \param player    The player index.
/// \param synchro   The synchronisation mode.
/// \param host      The hostname or IP address of the master (-1 if none).
/// \param channel   The channel number on the host (-1 if none).
///
/// \called-by  synchros_selection_changed_cb()
///             channels_selection_changed_cb()
///
function set_synchro(player, synchro, host, channel)
{
	var line = "action/set-synchro.php?"
	         + "player=" + (player + window.current_group * window.nb_players)
	         + "&synchro=" + synchro 
	         + "&host=" + host
	         + "&channel=" + channel;

	var xhr  = get_xhr_object();
	if (xhr != null) {
		xhr.onreadystatechange = function() {
			if ((xhr.readyState == 4) && (xhr.status == 200))
				load_and_display_player(player, false);
		};
		xhr.open("GET", line, true);
		xhr.send();
	}
}



/// \brief Callback for changes in hosts menu.
///
/// \param player  - The player index.
///
/// ¢alled-by  player.php "onchange" event.
///
function hosts_selection_changed_cb(player)
{
	var host_index  = document.getElementById('master-host-' + player).selectedIndex;

	if (host_index < 0)
		return;

	if ((host_index == 0) && (window.nb_players * window.nb_groups > 1)) { // Local
		if (player == 0) {
			window.synchro_channel[player + window.nb_players * window.current_group] = window.nb_players * window.current_group + 1;
		} else {
			window.synchro_channel[player + window.nb_players * window.current_group] = window.nb_players * window.current_group;
		}
	} else {
		window.synchro_channel[player + window.nb_players * window.current_group] = 0;
	}
	fill_groups_list(player, true);
}



/// \brief Callback for changes in group menu selection.
///
/// \param player - The player index.
///
/// ¢alled-by  player.php "onchange" event.
///
function groups_selection_changed_cb(player)
{
	var grp = document.getElementById('master-group-' + player).selectedIndex;
	if (grp < 0)
		return;
	fill_channels_list(player, true);
}



/// \brief Callback for changes in channels menu.
///
/// \param player   The player index (starting from zero).
///
/// ¢alled-by  player.php "onchange" event.
///            fill_channels_list()
///
function channels_selection_changed_cb(player)
{
	set_synchro(player, PLAYER_SLAVE_MODE,
		document.getElementById('master-host-' + player).value,
		document.getElementById('master-channel-' + player).value);
}



/// \brief Callback for changes in sequences menu.
///
/// \param player   the player index (starting from zero).
///
/// ¢alled-by  player.php "onchange" event.
///
function sequences_selection_changed_cb(player)
{
	var filename = document.getElementById('sequence-' + player).value;
	var dev = player + window.current_group * window.nb_players;

	// Notify the embedded player.
	var xhr = get_xhr_object();
	if (xhr != null) {
		xhr.onreadystatechange = function()
		{
			if ((xhr.readyState == 4) && (xhr.status == 200))
				load_and_display_player(player, false);
		};
		xhr.open("GET", "action/set-sequence.php?player=" + dev + "&filename=" + filename, true);
		xhr.send();
	}
}



/// \brief Start playing a sequence.
///
/// \param player   The player index.
///
/// ¢alled-by  player.php "onclick" event.
///
function start_player(player)
{
	var xhr = get_xhr_object();
	var dev = player + window.current_group * window.nb_players;

	if (xhr != null) {
		xhr.onreadystatechange = function()
		{
			if ((xhr.readyState == 4) && (xhr.status == 200))
				load_and_display_player(player, true);
		};
		xhr.open("GET", "action/start-player.php?player=" + dev, true);
		xhr.send();
	}
}



/// \brief Pause a currently playing sequence.
///
/// \param player   The player index.
///
/// ¢alled-by  player.php "onclick" event.
///
function pause_player(player)
{
	var xhr = get_xhr_object();
	var dev = player + window.current_group * window.nb_players;

	if (xhr != null) {
		xhr.onreadystatechange = function()
		{
			if ((xhr.readyState == 4) && (xhr.status == 200))
				load_and_display_player(player, true);
		};
		xhr.open("GET", "action/pause-player.php?player=" + dev, true);
		xhr.send();
	}
}



/// \brief Stop playing a sequence.
///
/// \param player   The index of the player.
///
/// ¢alled-by  player.php "onclick" event.
///
function stop_player(player)
{
	var xhr = get_xhr_object();
	var dev = player + window.current_group * window.nb_players;

	if (xhr != null) {
		xhr.onreadystatechange = function()
		{
			if ((xhr.readyState == 4) && (xhr.status == 200))
				load_and_display_player(player, true);
		};
		xhr.open("GET", "action/stop-player.php?player=" + dev, true);
		xhr.send();
	}
}






