// ------------------------------------------------------
// Device_canvas class implementation.
//
// Copyright (C) 2021 Logilin (ingenierie@logilin.fr)
//
// ------------------------------------------------------


function Device_canvas(num)
{
	this.number =  num;

	this.canvas = document.getElementById("dev-cnv-" + num);
	this.canvas.dev_cnv = this;

	this.text = document.getElementById("dev-txt-" + num);
	this.text.dev_cnv = this;
	this.text.addEventListener('mousedown', Device_canvas_mouse_down_cb);

	this.background = document.getElementById("dev-bkg-" + num);

	this.value_label = document.getElementById("dev-value-" + num);
	this.name_label  = document.getElementById("dev-name-" + num);
	this.div = document.getElementById("dev-div-" + num);
	this.td =  document.getElementById("dev-td-" + num);


	this.device_type =  DEVICE_ATTENUATOR;

	this.selected =  false;
	this.sid =  0;
	this.sid_error =  0;

	this.lock_id = 0;

	this.value = 0.1;
	this.moved =  0;
	this.x_offset =  0;
	this.y_offset =  0;
	this.permission =  User_permissions[this.number];
	this.min_value = 0;
	this.max_value = 1;
	this.granularity = 1;
	this.display_factor = 1.0;

	this.col_width = 0;
	this.line_height = 0;
	this.nb_lines = 0;
	this.nb_cols = 0;

	this.last_click_time = 0;
}



Device_canvas.prototype.draw_status = function(value, from_hmi)
{
	var ctx;
	var fgcolor;

	try {
		ctx = this.canvas.getContext("2d");

		if (Switch_on_remote_position) {
			if (this.lock_id != 0) {
				// the device is locked...
				if (this.lock_id == this.sid) {
					// ... by ourself.
					this.background.style.backgroundColor = Backcolor_locked_by_ourself;
					this.canvas.fillStyle = Fillcolor_locked_by_ourself;
					this.canvas.strokeStyle = Drawcolor_locked_by_ourself;

				} else {
					// ... by someone else.
					this.background.style.backgroundColor = Backcolor_locked_by_someone_else;
					this.canvas.fillStyle = Fillcolor_locked_by_someone_else;
					this.canvas.strokeStyle = Drawcolor_locked_by_someone_else;
				}

			} else {
				// the device is free...
				if (User_permissions[Current_group * Devices_per_group + this.number] == 0) {
					// but not allowed for us.
					this.background.style.backgroundColor = Backcolor_free_but_not_allowed;
					this.canvas.fillStyle = Fillcolor_free_but_not_allowed;
					this.canvas.strokeStyle = Drawcolor_free_but_not_allowed;

				} else {
					// and allowed for us.
					this.background.style.backgroundColor = Backcolor_free_for_us;
					this.canvas.fillStyle = Fillcolor_free_for_us;
					this.canvas.strokeStyle = Drawcolor_free_for_us;
				}
			}
		} else {
			// Switch on 'Local',
			// every devices are forbidden on web hmi.
			this.background.style.backgroundColor = Backcolor_switch_on_local;
			this.canvas.fillStyle = Fillcolor_switch_on_local;
			this.canvas.strokeStyle = Drawcolor_switch_on_local;
		}


		if (from_hmi) {
			// Nothing from hmi is accepted on 'Local' position...
			if (! Switch_on_remote_position)
				return;
			// ...nor if the device is currently commanded by someone else
			if ((this.sid == 0) || (this.lock_id != this.sid))
				return;
		} else {
			// Nothing from local is accepted if canvas selected and moving.
			if ((this.selected) && (Switch_on_remote_position) && (Device_canvas_mouse_move != null))
				return;
		}

		if ((this.value == parseInt(value)) && (this.canvas.fillStyle == ctx.fillStyle))
			return;

		this.value = parseInt(value);

		if (this.device_type == DEVICE_ATTENUATOR)
			this.draw_attenuator_status(ctx);
		else if (this.device_type == DEVICE_SWITCH)
			this.draw_switch_status(ctx);
		else if (this.device_type == DEVICE_SPST)
			this.draw_spst_status(ctx);
		else
			this.value_label.innerHTML = "";

	} catch (err) {
		alert(err);
	}
}



Device_canvas.prototype.draw_attenuator_status = function(ctx)
{
	var y;
	var l;

	if (this.granularity < 50)
		this.value_label.innerHTML = (this.value * this.display_factor).toFixed(2);
	else
		this.value_label.innerHTML = (this.value * this.display_factor).toFixed(1);

	this.canvas.width = this.canvas.width; // Clear
	y = (this.canvas.height - 2 * Canvas_margin) * (1 - this.value / Attenuators_range) + Canvas_margin;
	l = this.canvas.width / 3;
	// Fill the triangle
	ctx.beginPath();
	ctx.moveTo(this.canvas.width / 2 + 2, y);
	ctx.lineTo(this.canvas.width / 2 + l + 2, y + l / 2);
	ctx.lineTo(this.canvas.width / 2 + l + 2, y - l / 2);
	ctx.lineTo(this.canvas.width / 2 + 2, y);
	ctx.fillStyle = this.canvas.fillStyle;
	ctx.fill();
	ctx.strokeStyle = this.canvas.strokeStyle;
	ctx.stroke();
}



Device_canvas.prototype.draw_switch_status = function(ctx)
{
	var dh = (this.canvas.height - 2.0 * Canvas_margin) / Throws_per_switch;

	var position = this.value;
	var col = Math.trunc(position / 8);
	var line = this.nb_lines - 1 - (position % 8);

	this.value_label.innerHTML = this.value + 1;

	this.canvas.width = this.canvas.width; // Clear

	ctx.beginPath();
	ctx.arc(this.col_width * (col + 0.5),
	        this.line_height * (line + 0.5),
	        this.col_width * 0.35,
	        0,
	        2 * Math.PI);
	ctx.fillStyle = this.canvas.fillStyle;
	ctx.fill();

	ctx.moveTo(this.col_width * (col + 0.75),
	           this.line_height * (line + 0.5));

	ctx.arc(this.col_width * (col + 0.5),
	        this.line_height * (line + 0.5),
	        this.col_width / 4,
	        0,
	        2 * Math.PI, true);

	ctx.strokeStyle = this.canvas.strokeStyle;
	ctx.stroke();
}



Device_canvas.prototype.draw_spst_status = function(ctx)
{

	var dh = (this.canvas.height - 2.0 * Canvas_margin);

	var position = this.value;
	var col = Math.trunc(position / 8);
	var line = position % 8;

	if (this.value == 0)
		this.value_label.innerHTML = "OFF";
	else
		this.value_label.innerHTML = "ON";

	this.canvas.width = this.canvas.width; // Clear

	var ctx = this.text.getContext("2d");
	if (this.value) {
		ctx.fillStyle = this.canvas.fillStyle;
	} else {
		ctx.fillStyle = this.background.style.backgroundColor;
	}
	ctx.beginPath();
	ctx.arc(this.col_width * 0.5,
	        this.line_height * 0.5,
	        this.col_width / 4,
	        0, 2 * Math.PI);
	ctx.fill();

	ctx.beginPath();
	ctx.arc(this.col_width * 0.5,
	        this.line_height * 0.5,
	        this.col_width / 4,
	        0,
	        2 * Math.PI, true);
	ctx.strokeStyle = this.canvas.strokeStyle;
	ctx.stroke();

	if (this.value) {
		var txt_ctx = this.text.getContext("2d");
		txt_ctx.beginPath();
		txt_ctx.textAlign = "center";
		txt_ctx.textBaseline = "middle";
		txt_ctx.fillStyle = Drawcolor_devices_background;
		txt_ctx.font = "bold  10px Arial";
		txt_ctx.fillText("ON", this.col_width * 0.5, this.line_height * 0.5);
		txt_ctx.stroke();
	}
}




Device_canvas.prototype.draw_background = function()
{
	this.name_label.width = 60;
	this.value_label.width = 60;
	this.background.width = 60;
	this.canvas.width = 60;
	this.text.width = 60;
	this.td.width = 60;
	this.div.width = 60;

	if (this.device_type == DEVICE_ATTENUATOR)
		this.draw_attenuator_background();

	if (this.device_type == DEVICE_SWITCH) {
		if (Throws_per_switch > 8) {
			this.name_label.width = 120;
			this.value_label.width = 120;
			this.background.width = 120;
			this.canvas.width = 120;
			this.text.width = 120;
			this.div.width = 120;
			this.td.width = 120;
		}
		this.draw_switch_background();
	}

	if (this.device_type == DEVICE_SPST) {
		this.draw_spst_background();
	}
}



Device_canvas.prototype.set_offsets = function()
{
	this.x_offset = this.canvas.getBoundingClientRect().left;
	this.y_offset = this.canvas.getBoundingClientRect().top;
}



Device_canvas.prototype.draw_attenuator_background = function()
{
	var bkg_ctx = this.background.getContext("2d");

	var nb_steps     = Attenuators_range * this.display_factor;
	var step_height  = (this.background.height - 2.0 * Canvas_margin) / nb_steps;
	var y;
	var lg;

	// Draw the axis.
	bkg_ctx.beginPath();
	bkg_ctx.moveTo(this.background.width / 2, Canvas_margin);
	bkg_ctx.lineTo(this.background.width / 2, this.background.height - Canvas_margin);

	// Graduations.
	for (var i = 0; i < nb_steps; i ++) {
		y = this.background.height - Canvas_margin - i * step_height;
		if ((i % 10) == 0)
			lg = this.background.width / 6;
		else
			lg = this.background.width / 10;
		bkg_ctx.moveTo(this.background.width / 2, y);
		bkg_ctx.lineTo(this.background.width / 2 + lg, y);
	}
	lg = this.background.width / 4;
	y = Canvas_margin;
	bkg_ctx.moveTo(this.background.width / 2 - lg, y);
	bkg_ctx.lineTo(this.background.width / 2 + lg, y);
	y = this.background.height - Canvas_margin;
	bkg_ctx.moveTo(this.background.width / 2 - lg, y);
	bkg_ctx.lineTo(this.background.width / 2 + lg, y);

	bkg_ctx.strokeStyle = Drawcolor_devices_background;
	bkg_ctx.stroke();
}



Device_canvas.prototype.draw_switch_background = function()
{
	var nb_throws;

	this.nb_cols = 1 + Math.trunc((Throws_per_switch - 1)/ 8);
	this.col_width = this.background.width  / this.nb_cols;

	this.nb_lines = Throws_per_switch;
	if (this.nb_lines > 8)
		this.nb_lines = 8;
	this.line_height = this.background.height / this.nb_lines;

	var bkg_ctx = this.background.getContext("2d");
	bkg_ctx.beginPath();
	bkg_ctx.strokeStyle = Drawcolor_devices_background;

	nb_throws = 0;
	for (var col = 0; col < this.nb_cols; col ++) {
		for (var line = this.nb_lines - 1; line >= 0; line --) {
			if (++ nb_throws > Throws_per_switch)
				break;

			bkg_ctx.moveTo(this.col_width * (col + 0.85),
			           this.line_height * (line + 0.5));
			bkg_ctx.arc(this.col_width * (col + 0.5),
			        this.line_height * (line + 0.5),
			        this.col_width * 0.35,
			        0,
			        2 * Math.PI);

			bkg_ctx.moveTo(this.col_width * (col + 0.75),
			           this.line_height * (line + 0.5));
			bkg_ctx.arc(this.col_width * (col + 0.5),
			        this.line_height * (line + 0.5),
			        this.col_width / 4,
			        0,
			        2 * Math.PI, true);

		}
	}
	bkg_ctx.stroke();

	var txt_ctx = this.text.getContext("2d");
	txt_ctx.beginPath();
	txt_ctx.textAlign = "center";
	txt_ctx.textBaseline = "middle";
	txt_ctx.fillStyle = Drawcolor_devices_background;
	txt_ctx.font = "bold " + (this.col_width / 4 + 1) + "px Arial";
	nb_throws = 0;
	for (var col = 0; col < this.nb_cols; col ++) {
		for (var line = this.nb_lines - 1; line >= 0; line --) {
			if (++ nb_throws > Throws_per_switch)
				break;
			txt_ctx.fillText(nb_throws,
			             this.col_width * (col + 0.5),
			             this.line_height * (line + 0.5));
		}
	}
	txt_ctx.stroke();
}



Device_canvas.prototype.draw_spst_background = function()
{
	var nb_throws;

	this.nb_cols = 1
	this.col_width  = this.background.width;

	this.nb_lines = 1;
	this.line_height = this.background.height;

	var bkg_ctx = this.background.getContext("2d");
	bkg_ctx.beginPath();

	bkg_ctx.strokeStyle = Drawcolor_devices_background;
	bkg_ctx.moveTo(this.col_width * 0.85,
	               this.line_height * 0.5);
	bkg_ctx.arc(this.col_width * 0.5,
	            this.line_height * 0.5,
	            this.col_width * 0.35,
	            0,
	            2 * Math.PI);
	bkg_ctx.stroke();
}



Device_canvas.prototype.click = function(x, y)
{
	const d = new Date();
	if ((d.getTime() - this.last_click_time) < 100) // 100 milliseconds
		return;

	this.last_click_time = d.getTime();

	if (this.device_type == DEVICE_ATTENUATOR) {
		this.attenuator_canvas_click(0, y  - this.canvas.getBoundingClientRect().top);
	} else if (this.device_type == DEVICE_SWITCH) {
		this.switch_canvas_click(x - this.canvas.getBoundingClientRect().left, y  - this.canvas.getBoundingClientRect().top);
	} else {
		this.spst_canvas_click(0, y  - this.canvas.getBoundingClientRect().top);
	}
}



Device_canvas.prototype.attenuator_canvas_click = function(x, y)
{
	if (this.selected)
		this.unselect(true);
	else
		this.select();

	this.draw_status(this.value, true, Switch_on_remote_position);
}



Device_canvas.prototype.switch_canvas_click = function(x, y)
{
	var dist;
	var line;

	if (this.selected) {
		var _throw = 0;
		var _found = 0;
		for (var col = 0; col < this.nb_cols; col ++) {
			for (line = this.nb_lines - 1; line >= 0; line --) {
				if (++_throw > Throws_per_switch)
					break;
				dist = (x - this.col_width * (col + 0.5)) * (x - this.col_width * (col + 0.5))
				         + (y - this.line_height * (line + 0.5)) * (y - this.line_height * (line + 0.5));
				if (dist < (this.col_width * this.col_width) / 4) {
					_found = 1;
					break;
				}
			}
			if (line >= 0)
				break;
		}
		if (!_found) {
			this.unselect(true);
		} else {
			this.draw_status(_throw - 1, true, Switch_on_remote_position);
			set_devices();
		}
	} else {
		this.select();
	}
}



Device_canvas.prototype.spst_canvas_click = function(x, y)
{
	var distance;

	if (this.selected) {
		distance = (y - this.line_height *  0.5) * (y - this.line_height *  0.5);
		if (distance < (this.col_width * this.col_width) / 4) {
			this.draw_status(1 - this.value, true);
			set_devices();
		} else {
			this.unselect(true);
		}
	} else {
		this.select();
	}
}



Device_canvas.prototype.select = function()
{
	if (! this.selected) {
		var dev_num = this.number + Devices_per_group * Current_group;
		var cnv_dev = this;
		if (this.sid == 0) {
			jQuery.ajax({
				type: "GET",
				url:"cgi-bin/lock-device?" + dev_num,
				dataType: 'text',
				async: true,
				success: function(data, textStatus, jqXHR) {
					if (cnv_dev.sid == 0) {
						cnv_dev.sid = parseInt(data);
						cnv_dev.selected = (cnv_dev.sid != 0);
						cnv_dev.refresh_device_lock();
					}
				}
			});
		}
	}
}



Device_canvas.prototype.unselect = function(asynchronous)
{
	if (this.selected) {
		var dev_num = this.number + Devices_per_group * Current_group;
		jQuery.ajax({
			type: "GET",
			url:"cgi-bin/unlock-device?" + dev_num + "," + this.sid,
			dataType: 'text',
			async: asynchronous,
		});
		this.sid = 0;
		this.selected = false;
	}
}



Device_canvas.prototype.refresh_device_lock = function()
{
	if (this.selected) {
		var num = this.number + Devices_per_group * Current_group;
		jQuery.ajax({
			type: "GET",
			url:"cgi-bin/refresh-device-lock?" + num + "," + this.sid,
			dataType: 'text',
			async: true,
			complete: function(jqXHR, textStatus) {	}
		});
	}
}



Device_canvas.prototype.hide = function()
{
	this.value_label.style.display = 'none';
	this.name_label.style.display = 'none';
	this.td.style.display = 'none';
}



Device_canvas.prototype.show = function()
{
	this.value_label.style.display = '';
	this.name_label.style.display = '';
	this.td.style.display = '';
}

